// Global app state
const AppState = {
    currentTheme: localStorage.getItem('theme') || 'light',
    isGenerating: false,
    generatedArticles: [],
    userSettings: {},
    notifications: []
};

// DOM Content Loaded
document.addEventListener('DOMContentLoaded', function() {
    initializeTheme();
    initializeTabs();
    initializeCollapsibles();
    initializeUserMenu();
    initializeRangeInputs();
    loadUserSettings();
});

// Theme Management
function initializeTheme() {
    const themeToggle = document.getElementById('themeToggle');
    const html = document.documentElement;
    
    // Set initial theme
    html.setAttribute('data-theme', AppState.currentTheme);
    updateThemeIcon();
    
    if (themeToggle) {
        themeToggle.addEventListener('click', toggleTheme);
    }
}

function toggleTheme() {
    AppState.currentTheme = AppState.currentTheme === 'light' ? 'dark' : 'light';
    document.documentElement.setAttribute('data-theme', AppState.currentTheme);
    localStorage.setItem('theme', AppState.currentTheme);
    updateThemeIcon();
}

function updateThemeIcon() {
    const themeToggle = document.getElementById('themeToggle');
    if (themeToggle) {
        const icon = themeToggle.querySelector('.mdi');
        if (icon) {
            icon.className = AppState.currentTheme === 'light' 
                ? 'mdi mdi-weather-night' 
                : 'mdi mdi-weather-sunny';
        }
    }
}

// Tab Management
function initializeTabs() {
    const tabButtons = document.querySelectorAll('.tab-button');
    
    tabButtons.forEach(button => {
        button.addEventListener('click', function() {
            const tabId = this.getAttribute('data-tab');
            switchTab(tabId);
        });
    });
}

function switchTab(tabId) {
    // Update tab buttons
    document.querySelectorAll('.tab-button').forEach(btn => {
        btn.classList.remove('active');
    });
    document.querySelector(`[data-tab="${tabId}"]`).classList.add('active');
    
    // Update tab panes
    document.querySelectorAll('.tab-pane').forEach(pane => {
        pane.classList.remove('active');
    });
    document.getElementById(`${tabId}-tab`).classList.add('active');
}

// Collapsible Sections
function initializeCollapsibles() {
    document.querySelectorAll('.collapsible-header').forEach(header => {
        header.addEventListener('click', function() {
            const collapsible = this.parentElement;
            collapsible.classList.toggle('expanded');
            
            const icon = this.querySelector('.mdi');
            if (icon) {
                icon.className = collapsible.classList.contains('expanded') 
                    ? 'mdi mdi-chevron-up' 
                    : 'mdi mdi-chevron-down';
            }
        });
    });
}

// Range Inputs
function initializeRangeInputs() {
    // Initialize creativity range
    const creativitySlider = document.getElementById('creativity');
    const creativityValue = document.getElementById('creativityValue');
    if (creativitySlider && creativityValue) {
        creativitySlider.addEventListener('input', function() {
            creativityValue.textContent = this.value + '%';
        });
    }

    // Initialize research range
    const researchSlider = document.getElementById('research');
    const researchValue = document.getElementById('researchValue');
    if (researchSlider && researchValue) {
        researchSlider.addEventListener('input', function() {
            researchValue.textContent = this.value + '%';
        });
    }
}

// User Menu
function initializeUserMenu() {
    const userButton = document.querySelector('.user-button');
    const userDropdown = document.querySelector('.user-dropdown');
    
    if (userButton && userDropdown) {
        userButton.addEventListener('click', function(e) {
            e.stopPropagation();
            userDropdown.classList.toggle('show');
        });
        
        // Close dropdown when clicking outside
        document.addEventListener('click', function() {
            userDropdown.classList.remove('show');
        });
    }
}

// User Settings Management
async function loadUserSettings() {
    try {
        const response = await fetch('get_user_settings.php');
        const data = await response.json();
        
        if (data.success) {
            AppState.userSettings = data.settings;
            
            // Update API key field if it exists
            const apiKeyInput = document.getElementById('apiKey');
            if (apiKeyInput && data.settings.ai_api_key) {
                // Remove the AIzaSy prefix for display
                apiKeyInput.value = data.settings.ai_api_key.substring(6);
            }
        }
    } catch (error) {
        console.warn('Failed to load user settings:', error);
    }
}

// API Key Management
async function saveAPIKey() {
    const apiKeyInput = document.getElementById('apiKey');
    const key = apiKeyInput.value.trim();
    
    if (!key) {
        showNotification('Please enter an API key', 'error');
        return;
    }

    // Add the AIzaSy prefix
    const fullKey = 'AIzaSy' + key;
    
    try {
        const response = await fetch('save_settings.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'save_api_key',
                api_key: fullKey
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification('API key saved successfully!', 'success');
            AppState.userSettings.ai_api_key = fullKey;
        } else {
            showNotification(data.message || 'Failed to save API key', 'error');
        }
    } catch (error) {
        showNotification('Failed to save API key: ' + error.message, 'error');
    }
}

async function saveAPIKeys() {
    const apiKeyInput = document.getElementById('apiKey');
    const key = apiKeyInput.value.trim();
    
    if (!key) {
        showNotification('Please enter an API key', 'error');
        return;
    }

    // Add the AIzaSy prefix
    const fullKey = 'AIzaSy' + key;
    
    try {
        const response = await fetch('save_settings.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'save_api_key',
                api_key: fullKey
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification('API key saved successfully!', 'success');
            AppState.userSettings.ai_api_key = fullKey;
        } else {
            showNotification(data.message || 'Failed to save API key', 'error');
        }
    } catch (error) {
        showNotification('Failed to save API key: ' + error.message, 'error');
    }
}

// Notification System
function showNotification(message, type = 'info', duration = 5000) {
    // Create notifications container if it doesn't exist
    let container = document.getElementById('notificationsContainer');
    if (!container) {
        container = document.createElement('div');
        container.className = 'notifications-container';
        container.id = 'notificationsContainer';
        document.body.appendChild(container);
    }
    
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.innerHTML = `
        <div class="notification-icon">
            <i class="mdi mdi-${getNotificationIcon(type)}"></i>
        </div>
        <div class="notification-content">
            <p>${message}</p>
        </div>
        <button class="notification-close" onclick="this.parentElement.remove()">
            <i class="mdi mdi-close"></i>
        </button>
    `;
    
    container.appendChild(notification);
    
    // Auto-remove after duration
    setTimeout(() => {
        if (notification.parentElement) {
            notification.remove();
        }
    }, duration);
    
    return notification;
}

function getNotificationIcon(type) {
    const icons = {
        'success': 'check-circle',
        'error': 'alert-circle',
        'warning': 'alert',
        'info': 'information'
    };
    return icons[type] || 'information';
}

// Load notifications from server
async function loadNotifications() {
    try {
        const response = await fetch('get_notifications.php');
        const data = await response.json();
        
        if (data.success && data.notifications) {
            data.notifications.forEach(notification => {
                showNotification(notification.message, notification.type);
            });
        }
    } catch (error) {
        console.warn('Failed to load notifications:', error);
    }
}

// Utility Functions
function formatWordCount(count) {
    return count.toLocaleString();
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

function throttle(func, limit) {
    let inThrottle;
    return function(...args) {
        if (!inThrottle) {
            func.apply(this, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    }
}

// File Export Functions
function exportAsDoc(content, filename) {
    // Enhanced HTML content with better styling
    const htmlContent = `
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>${filename}</title>
            <style>
                body { 
                    font-family: 'Calibri', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; 
                    line-height: 1.6; 
                    margin: 2rem;
                    color: #333;
                    max-width: 800px;
                    margin-left: auto;
                    margin-right: auto;
                }
                h1, h2, h3, h4, h5, h6 { 
                    color: #2c3e50;
                    margin-top: 1.5em;
                    margin-bottom: 0.5em;
                }
                h1 { 
                    font-size: 28px; 
                    border-bottom: 3px solid #3498db;
                    padding-bottom: 10px;
                }
                h2 { font-size: 24px; }
                h3 { font-size: 20px; }
                h4 { font-size: 18px; }
                h5 { font-size: 16px; }
                h6 { font-size: 14px; }
                p { 
                    margin-bottom: 1em;
                    text-align: justify;
                }
                strong { 
                    font-weight: bold;
                    color: #2c3e50;
                }
                em { 
                    font-style: italic;
                    color: #7f8c8d;
                }
                u { 
                    text-decoration: underline;
                }
                ul, ol {
                    margin: 1em 0;
                    padding-left: 2em;
                }
                li {
                    margin-bottom: 0.5em;
                }
                blockquote {
                    border-left: 4px solid #3498db;
                    padding: 1em 1.5em;
                    margin: 1.5em 0;
                    background: #f8f9fa;
                    font-style: italic;
                }
                code {
                    background: #f8f9fa;
                    padding: 0.2em 0.4em;
                    border-radius: 3px;
                    font-family: 'Courier New', monospace;
                    font-size: 0.9em;
                }
                pre {
                    background: #2c3e50;
                    color: #ecf0f1;
                    padding: 1.5em;
                    border-radius: 5px;
                    overflow-x: auto;
                    margin: 1.5em 0;
                }
                pre code {
                    background: none;
                    padding: 0;
                    color: inherit;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                    margin: 1.5em 0;
                }
                th, td {
                    padding: 0.75em;
                    text-align: left;
                    border-bottom: 1px solid #ddd;
                }
                th {
                    background: #f8f9fa;
                    font-weight: bold;
                }
                .meta-info {
                    background: #f8f9fa;
                    padding: 1em;
                    border-radius: 5px;
                    margin-bottom: 2em;
                    font-size: 0.9em;
                    color: #7f8c8d;
                }
            </style>
        </head>
        <body>
            <div class="meta-info">
                <strong>Document:</strong> ${filename}<br>
                <strong>Generated:</strong> ${new Date().toLocaleString()}<br>
                <strong>Word Count:</strong> ${content.split(/\s+/).length}
            </div>
            ${formatContentForExport(content)}
        </body>
        </html>
    `;
    
    const blob = new Blob([htmlContent], { type: 'application/msword' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `${filename.replace(/[^a-z0-9]/gi, '_')}.doc`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
}

function formatContentForExport(content) {
    // Enhanced formatting for export
    return content
        .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
        .replace(/\*(.*?)\*/g, '<em>$1</em>')
        .replace(/__(.*?)__/g, '<u>$1</u>')
        .replace(/\n(# .*?)(?=\n|$)/g, '\n<h1>$1</h1>')
        .replace(/\n(## .*?)(?=\n|$)/g, '\n<h2>$1</h2>')
        .replace(/\n(### .*?)(?=\n|$)/g, '\n<h3>$1</h3>')
        .replace(/\n(#### .*?)(?=\n|$)/g, '\n<h4>$1</h4>')
        .replace(/\n(- .*?)(?=\n|$)/g, '\n<li>$1</li>')
        .replace(/(<li>.*<\/li>)/s, '<ul>$1</ul>')
        .replace(/\n(\d+\. .*?)(?=\n|$)/g, '\n<li>$1</li>')
        .replace(/(<li>.*<\/li>)/s, '<ol>$1</ol>')
        .replace(/\n&gt; (.*?)(?=\n|$)/g, '\n<blockquote>$1</blockquote>')
        .replace(/`(.*?)`/g, '<code>$1</code>')
        .replace(/```([\s\S]*?)```/g, '<pre><code>$1</code></pre>')
        .replace(/\n\n/g, '</p><p>')
        .replace(/^/, '<p>')
        .replace(/$/, '</p>');
}

function copyToClipboard(text) {
    // Create a temporary textarea element
    const textarea = document.createElement('textarea');
    textarea.value = text;
    textarea.style.position = 'fixed';
    textarea.style.opacity = '0';
    document.body.appendChild(textarea);
    
    // Select and copy the text
    textarea.select();
    textarea.setSelectionRange(0, 99999); // For mobile devices
    
    try {
        const successful = document.execCommand('copy');
        document.body.removeChild(textarea);
        
        if (successful) {
            showNotification('Content copied to clipboard!', 'success');
            return true;
        } else {
            showNotification('Failed to copy content', 'error');
            return false;
        }
    } catch (err) {
        document.body.removeChild(textarea);
        
        // Fallback for modern browsers
        if (navigator.clipboard && navigator.clipboard.writeText) {
            navigator.clipboard.writeText(text).then(() => {
                showNotification('Content copied to clipboard!', 'success');
            }).catch(() => {
                showNotification('Failed to copy content', 'error');
            });
        } else {
            showNotification('Failed to copy content', 'error');
        }
        return false;
    }
}

// Content Formatting
function formatContentForDisplay(content) {
    // Convert markdown-like formatting to HTML with enhanced styling
    return content
        .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
        .replace(/\*(.*?)\*/g, '<em>$1</em>')
        .replace(/__(.*?)__/g, '<u>$1</u>')
        .replace(/\n(# .*?)(?=\n|$)/g, '\n<h1>$1</h1>')
        .replace(/\n(## .*?)(?=\n|$)/g, '\n<h2>$1</h2>')
        .replace(/\n(### .*?)(?=\n|$)/g, '\n<h3>$1</h3>')
        .replace(/\n(#### .*?)(?=\n|$)/g, '\n<h4>$1</h4>')
        .replace(/\n(- .*?)(?=\n|$)/g, '\n<li>$1</li>')
        .replace(/(<li>.*<\/li>)/s, '<ul>$1</ul>')
        .replace(/\n(\d+\. .*?)(?=\n|$)/g, '\n<li>$1</li>')
        .replace(/(<li>.*<\/li>)/s, '<ol>$1</ol>')
        .replace(/\n&gt; (.*?)(?=\n|$)/g, '\n<blockquote>$1</blockquote>')
        .replace(/`(.*?)`/g, '<code>$1</code>')
        .replace(/```([\s\S]*?)```/g, '<pre><code>$1</code></pre>')
        .replace(/\n\n/g, '</p><p>')
        .replace(/^/, '<p>')
        .replace(/$/, '</p>');
}

// Progress Tracking
class ProgressTracker {
    constructor(total) {
        this.total = total;
        this.current = 0;
        this.callbacks = [];
    }
    
    update(progress) {
        this.current = progress;
        this.callbacks.forEach(callback => callback(this.getPercentage()));
    }
    
    increment() {
        this.current++;
        this.callbacks.forEach(callback => callback(this.getPercentage()));
    }
    
    getPercentage() {
        return (this.current / this.total) * 100;
    }
    
    onProgress(callback) {
        this.callbacks.push(callback);
    }
}

// Modal Management
function showModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
    }
}

function hideModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto';
    }
}

// Settings Management
function showSettings() {
    const modal = document.getElementById('settingsModal');
    const apiKeyInput = document.getElementById('settingsApiKey');
    
    if (modal && apiKeyInput) {
        // Pre-fill with current API key (without prefix)
        if (AppState.userSettings.ai_api_key) {
            apiKeyInput.value = AppState.userSettings.ai_api_key.substring(6);
        }
        showModal('settingsModal');
    }
}

function hideSettings() {
    hideModal('settingsModal');
}

async function saveSettings() {
    const apiKeyInput = document.getElementById('settingsApiKey');
    const key = apiKeyInput.value.trim();
    
    if (!key) {
        showNotification('Please enter an API key', 'error');
        return;
    }

    // Add the AIzaSy prefix
    const fullKey = 'AIzaSy' + key;
    
    try {
        const response = await fetch('save_settings.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'save_api_key',
                api_key: fullKey
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification('Settings saved successfully!', 'success');
            AppState.userSettings.ai_api_key = fullKey;
            hideSettings();
        } else {
            showNotification(data.message || 'Failed to save settings', 'error');
        }
    } catch (error) {
        showNotification('Failed to save settings: ' + error.message, 'error');
    }
}

// Support Functions
function supportDeveloper() {
    window.open('https://quillfy.com/', '_blank');
}

function getHelp() {
    window.open('https://quillfy.com/', '_blank');
}

// Keyboard Shortcuts
document.addEventListener('keydown', function(e) {
    // Ctrl/Cmd + S to save
    if ((e.ctrlKey || e.metaKey) && e.key === 's') {
        e.preventDefault();
        if (typeof saveAPIKey === 'function') {
            saveAPIKey();
        }
    }
    
    // Ctrl/Cmd + Enter to generate all
    if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
        e.preventDefault();
        if (typeof generateAll === 'function') {
            generateAll();
        }
    }
    
    // Escape to close modals
    if (e.key === 'Escape') {
        const openModal = document.querySelector('.modal[style*="display: flex"]');
        if (openModal) {
            hideModal(openModal.id);
        }
        
        // Close dropdowns
        const openDropdown = document.querySelector('.user-dropdown.show');
        if (openDropdown) {
            openDropdown.classList.remove('show');
        }
    }
});

// Error Handling
window.addEventListener('error', function(e) {
    console.error('Global error:', e.error);
    showNotification('An unexpected error occurred. Please try again.', 'error');
});

window.addEventListener('unhandledrejection', function(e) {
    console.error('Unhandled promise rejection:', e.reason);
    showNotification('Thanks for using Quillfy AI Bulk Writer.', 'info');
});

// Performance Monitoring
const performanceMetrics = {
    startTime: null,
    endTime: null,
    
    startTimer() {
        this.startTime = performance.now();
    },
    
    endTimer() {
        this.endTime = performance.now();
        return this.endTime - this.startTime;
    },
    
    logMetric(name, value) {
        console.log(`[Performance] ${name}: ${value}ms`);
    }
};

// Initialize performance monitoring for page load
performanceMetrics.startTimer();
window.addEventListener('load', function() {
    const loadTime = performanceMetrics.endTimer();
    performanceMetrics.logMetric('Page Load Time', loadTime);
});

// Utility function to check if element is in viewport
function isInViewport(element) {
    const rect = element.getBoundingClientRect();
    return (
        rect.top >= 0 &&
        rect.left >= 0 &&
        rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
        rect.right <= (window.innerWidth || document.documentElement.clientWidth)
    );
}

// Lazy loading for images
function initializeLazyLoading() {
    const lazyImages = document.querySelectorAll('img[data-src]');
    
    const imageObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const img = entry.target;
                img.src = img.dataset.src;
                img.removeAttribute('data-src');
                imageObserver.unobserve(img);
            }
        });
    });
    
    lazyImages.forEach(img => imageObserver.observe(img));
}

// Initialize lazy loading when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeLazyLoading);
} else {
    initializeLazyLoading();
}

// Export utility functions for use in other files
window.AppUtils = {
    showNotification,
    formatWordCount,
    debounce,
    throttle,
    exportAsDoc,
    copyToClipboard,
    formatContentForDisplay,
    formatContentForExport,
    performanceMetrics,
    isInViewport
};

// Compatibility with older browsers
if (!String.prototype.trim) {
    String.prototype.trim = function() {
        return this.replace(/^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/g, '');
    };
}

// Polyfill for Element.closest
if (!Element.prototype.closest) {
    Element.prototype.closest = function(s) {
        var el = this;
        if (!document.documentElement.contains(el)) return null;
        do {
            if (el.matches(s)) return el;
            el = el.parentElement || el.parentNode;
        } while (el !== null && el.nodeType === 1);
        return null;
    };
}

// Polyfill for Element.matches
if (!Element.prototype.matches) {
    Element.prototype.matches = 
        Element.prototype.matchesSelector || 
        Element.prototype.mozMatchesSelector ||
        Element.prototype.msMatchesSelector || 
        Element.prototype.oMatchesSelector || 
        Element.prototype.webkitMatchesSelector ||
        function(s) {
            var matches = (this.document || this.ownerDocument).querySelectorAll(s),
                i = matches.length;
            while (--i >= 0 && matches.item(i) !== this) {}
            return i > -1;            
        };
}

console.log('Bulk AI Writer - Script initialized successfully');